<?php

// XXX ajax requests without valid arguments return last 20 results

require_once "./functions.php";

$SEARCHABLE_FIELDS = ['items.type', 'brand', 'size', 'color', 'items.description', 'location'];
$SORTABLE_FIELDS = [
	'items.date' => 'Fecha',
	'items.id' => 'Número',
	'items.location' => 'Ubicación',
	'price' => 'Precio'
];
$DEFAULT_SORT_BY = 'items.date';

$PAGE_SIZE = 10;

// filter parameters
$item_ids = get_string_or_null('item_ids');
$only_unsold = get_string_or_null('only_unsold') === '1';
$sale_id = get_string_or_null('sale_id');
$date_from = get_string_or_null('date_from');
$date_to = get_string_or_null('date_to');
$location = get_string_or_null('location');
$price_from = get_string_or_null('price_from');
$price_to = get_string_or_null('price_to');
$search_text = get_string_or_null('search_text');
// sorting parameters
$sort_by = get_string_or_null('sort_by');
$sort_dir = get_string_or_null('sort_dir');
$page = get_string_or_null('page');
$pagesize = get_string_or_null('pagesize');
// simplify the table for displaying in sales screen
$preview = get_string_or_null('preview') === '1';
// serialize records as json, used in ajax requests
$json_format = get_string_or_null('json') === '1';

// build sql WHERE statement
$sql_where = ' TRUE ';
$bind_types = '';
$bind_values = [];

// for each valid filter append its param
$query_string = '';

if ($item_ids) {
	$item_ids = trim($item_ids);
	$item_ids = trim($item_ids, ',');
	if (preg_match('/^(\s*\d+\s*)(,\s*\d+\s*)*?$/', $item_ids)) {
		$placeholders = '';
		foreach (explode(',', $item_ids) as $token) {
			$item_id = trim($token);
			$placeholders .= '?,';
			$bind_types .= 'i';
			$bind_values[] = $item_id;
		}
		$placeholders = rtrim($placeholders, ',');
		$sql_where .= " AND items.id IN ( $placeholders ) ";
		$query_string .= "&item_ids=$item_ids";
	}
}

if ($date_from) {
	$sql_where .= ' AND DATE(items.date) >= DATE(?) ';
	$bind_types .= 's';
	$bind_values[] = $date_from;
	$query_string .= "&date_from=$date_from";
}

if ($date_to) {
	$sql_where .= ' AND DATE(items.date) <= DATE(?) ';
	$bind_types .= 's';
	$bind_values[] = $date_to;
	$query_string .= "&date_to=$date_to";
}

if ($only_unsold) {
	$sql_where .= ' AND sale_id IS NULL ';
	$query_string .= '&only_unsold=1';
}

if ($sale_id) {
	$sql_where .= ' AND sale_id = ? ';
	$bind_types .= 'i';
	$bind_values[] = $sale_id;
	$query_string .= "&sale_id=$sale_id";
}

if ($location) {
	$sql_where .= ' AND items.location LIKE ? ';
	$bind_types .= 's';
	$bind_values[] = "%$location%";
	$query_string .= "&location=" . urlencode($location);
}

if ($price_from) {
	$sql_where .= ' AND price >= ? ';
	$bind_types .= 'i';
	$bind_values[] = $price_from;
	$query_string .= "&price_from=$price_from";
}

if ($price_to) {
	$sql_where .= ' AND price <= ? ';
	$bind_types .= 'i';
	$bind_values[] = $price_to;
	$query_string .= "&price_to=$price_to";
}

if ($search_text) {
	foreach (tokenize_string($search_text) as $token) {
		$sql_where .= ' AND (FALSE ';
		foreach ($SEARCHABLE_FIELDS as $field) {
			$sql_where .= " OR $field LIKE ? ";
			$bind_types .= 's';
			$bind_values[] = "%$token%";
		}		
		$sql_where .= ' ) ';
	}
	$query_string .= "&search_text=" . urlencode($search_text);
}

// validate sort parameters
$sort_by = (isset($SORTABLE_FIELDS[$sort_by]) ? $sort_by : $DEFAULT_SORT_BY);
$sort_dir = ($sort_dir === 'ASC' ? 'ASC' : 'DESC');
$query_string .= "&sort_by=$sort_by&sort_dir=$sort_dir";

// pagination
if (is_numeric($pagesize)) {
	$pagesize = intval($pagesize);
	if ($pagesize <= 0)
		$pagesize = $PAGE_SIZE;
} else
	$pagesize = $PAGE_SIZE;

$query_string .= "&pagesize=$pagesize";

$conn = new_db_connection();
$sql = "SELECT COUNT(*) AS count FROM items WHERE $sql_where";
$record_count = select_query($conn, $sql, $bind_types, $bind_values)[0]['count'];
$total_pages = ceil($record_count / $pagesize);

if (is_numeric($page)) {
	$page = intval($page);
	if ($page < 1)
		$page = 1;
	else if ($page > $total_pages)
		$page = $total_pages;
} else
	$page = 1;

$offset = ($page - 1) * $pagesize;

$first_page_link = "$_SERVER[PHP_SELF]?$query_string&page=1";
$last_page_link = "$_SERVER[PHP_SELF]?$query_string&page=$total_pages";
$next_page_link = "$_SERVER[PHP_SELF]?$query_string&page=" . ($page + 1);
$prev_page_link = "$_SERVER[PHP_SELF]?$query_string&page=" . ($page - 1);

// fetch records
$sql = <<<SQL
	SELECT
		items.id AS id,
		items.date AS date,
		items.type AS type,
		brand,
		size,
		color,
		location,
		price,
		items.description AS description,
		sale_id,
		sales.date AS sale_date,
		sales.type AS sale_type,
		sales.description AS sale_description
	FROM items
	LEFT JOIN sales on sales.id = sale_id
	WHERE $sql_where
	ORDER BY $sort_by $sort_dir
	LIMIT $pagesize OFFSET $offset
SQL;

$items = select_query($conn, $sql, $bind_types, $bind_values);

if ($json_format) {
	header('Content-Type: application/json');
	echo json_encode($items);
	exit;
}

?>
<!DOCTYPE html>
<html lang="es">
<head>
	<meta charset="utf-8" />
	<title>Predas - yendovintage</title>
	<meta name="viewport" content="width=device-width; initial-scale=1.0;" />
	<link rel="stylesheet" href="css/jquery-ui.css" />
	<link rel="stylesheet" href="css/styles.css" />
</head>
<body>
<?php if (!$preview): ?>
	<?php require_once 'navbar.php' ?>

	<h2>Prendas</h2>
	<form class="searchForm">
		<div class="filter">
			Fecha desde: <input type="text" name="date_from" id="date_from">
		</div>
		<div class="filter">
			Fecha hasta: <input type="text" name="date_to" id="date_to">
		</div>
		<div class="filter">
			Número: <input type="text" name="item_ids">
		</div>
		<div class="filter">
			Ubicación: <input type="text" name="location">
		</div>
		<div class="filter">
			<label for="price_for">Precio desde:</label>
			<input type="number" name="price_from" id="price_from">
		</div>
		<div class="filter">
			<label for="price_to">Precio hasta:</label>
			<input type="number" name="price_to" id="price_to">
		</div>
		<div class="filter">
			Buscar: <input type="text" name="search_text">
		</div>
		<div class="filter">
			Solo prendas sin vender: <input type="checkbox" name="only_unsold" value="1">
		</div>
		<div class="filter">
			<label for="sort_by">Ordenar por:</label>
			<select name="sort_by" id="sort_by">
			<?php foreach ($SORTABLE_FIELDS as $value => $label): ?>
				<option value="<?= $value ?>"><?= $label ?></option>
			<?php endforeach ?>
			</select>
		</div>
		<div class="filter">
			<label for="sort_dir">Orden:</label>
			<select name="sort_dir" id="sort_dir">
				<option>DESC</option>
				<option>ASC</option>
			</select>
		</div>
		<div class="filter">
			<label for="pagesize">Tamaño de página:</label>
			<select name="pagesize" id="pagesize">
				<option>10</option>
				<option>20</option>
				<option>30</option>
				<option>50</option>
				<option>100</option>
			</select>
		</div>
		<div class="filter">
			<input type="submit" value="BUSCAR" />
		</div>
	</form>
	<p style="font-weight: bold">Cantidad de resultados: <?= $record_count ?></p>
<?php endif ?>
	<table border="1">
		<tr>
			<th>Número</th>
			<th>Fecha ingreso</th>
			<th>Tipo</th>
			<th>Marca</th>
			<th>Talle</th>
			<th>Color</th>
			<th>Precio</th>
			<th>Ubicación</th>
			<th>Descripción</th>
			<th>Fecha vendido</th>
			<th>Tipo venta</th>
		<?php if (!$preview && check_admin()): ?>
			<th></th>
		<?php endif ?>	
		</tr>
		<?php foreach ($items as $item): ?>
			<tr style="background-color: <?= $item['sale_id'] && !$preview ? 'lightgray' : 'default' ?>">
				<td style="text-align: right"><?= $item['id'] ?></td>
				<td><?= $item['date'] ?></td>
				<td><?= htmlentities($item['type']) ?></td>
				<td><?= htmlentities($item['brand']) ?></td>
				<td><?= htmlentities($item['size']) ?></td>
				<td><?= htmlentities($item['color']) ?></td>
				<td style="text-align: right"><?= format_money($item['price']) ?></td>
				<td><?= htmlentities($item['location']) ?></td>
				<td><?= htmlentities($item['description'] ?? '') ?></td>
				<td><?= $item['sale_date'] ?></td>
				<td><?= htmlentities($item['sale_type'] ?? '') ?></td>
			<?php if (!$preview && check_admin()): ?>
				<td>
				<?php if (!$item['sale_id']): ?>
					<a href="edit_item.php?id=<?= $item['id'] ?>">Editar</a>
				<?php endif ?>
				</td>
			<?php endif ?>
			</tr>
		<?php endforeach ?>
	</table>
<?php if (!$preview): ?>
	<div class="pagination">
		<a class="pageLink" href="<?= $first_page_link ?>">&lt;&lt;&lt; Primera</a>
		<a class="pageLink" href="<?= $prev_page_link ?>">&lt; Anterior</a>
		<span class="pageCount"><?= $page ?> / <?= $total_pages ?></span>
		<a class="pageLink" href="<?= $next_page_link ?>">Siguiente &gt;</a>
		<a class="pageLink" href="<?= $last_page_link ?>">Última &gt;&gt;&gt;</a>
	</div>
	<script src="js/jquery-3.7.1.js"></script>
	<script src="js/jquery-ui.js"></script>
	<script src="js/datepicker-es.js"></script>
	<script>
		$(function(){
			$("#date_from, #date_to")
				.datepicker($.datepicker.regional["es"])
				.datepicker("option","dateFormat","yy-mm-dd");
		})
	</script>
<?php endif ?>
</body>
</html>
